/*
 * CG_dispatcher_user.c
 * 
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of the project's author nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/*
This Uart driver uses the SO10 and SI10 which are connected to Pins 7 &8 of Header 1
*******************************************************************************
** Include files
*******************************************************************************
*/
#include "r_cg_macrodriver.h"
#include "r_cg_serial.h"
#include "r_cg_userdefine.h"
//#include "r_cg_system.h"
#include "CG_dispatcher_user.h"

/*
*******************************************************************************
**  Global define
*******************************************************************************
*/
volatile uint8_t	*gpUart1TxAddress;	/* uart1 transmit buffer address */
volatile uint16_t	gUart1TxCnt;		/* uart1 transmit data number */
volatile uint8_t	*gpUart1RxAddress;	/* uart1 receive buffer address */
volatile uint16_t	gUart1RxCnt;		/* uart1 receive data number */

volatile uint16_t	gUart1RxLen;		/* uart1 receive data length */
volatile uint8_t uart_have_cmd = 0;
uint8_t g_ucUARTBuffer[USART_IF_BUFFER];
/* Start user code for global. Do not edit comment generated here */
/* UART related defines */

static void UartWaitForTxFinish(void);


/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function is INTSR1 interrupt service routine.
**
**  Parameters:
**	None
**
**  Returns:
**	None
**
**-----------------------------------------------------------------------------
*/
#pragma vector = INTSR1_vect
__interrupt void MD_INTSR1(void)
{
	uint8_t rx_data;

	rx_data = RXD1;
	if (gUart1RxLen > gUart1RxCnt)
	{
		*gpUart1RxAddress = rx_data;
		gpUart1RxAddress++;
		gUart1RxCnt++;
	}

}
/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function is INTSRE1 interrupt service routine.
**
**  Parameters:
**	None
**
**  Returns:
**	None
**
**-----------------------------------------------------------------------------
*/
#pragma vector = INTSRE1_vect
__interrupt void MD_INTSRE1(void)
{
	uint8_t err_type;

	*gpUart1RxAddress = RXD1;
	err_type = (uint8_t)(SSR03 & 0x0007U);
	SIR03 = (uint16_t)err_type;
}
/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function is INTST1 interrupt service routine.
**
**  Parameters:
**	None
**
**  Returns:
**	None
**
**-----------------------------------------------------------------------------
*/
#pragma vector = INTST1_vect
__interrupt void MD_INTST1(void)
{
	if (gUart1TxCnt > 0U)
	{
		TXD1 = *gpUart1TxAddress;
		gpUart1TxAddress++;
		gUart1TxCnt--;
	}
}

/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function starts the UART1 module operation.
**
**  Parameters:
**	None
**
**  Returns:
**	None
**
**-----------------------------------------------------------------------------
*/
void UART1_Start(void)
{
	STIF1 = 0U;	/* clear INTST1 interrupt flag */
	STMK1 = 0U;	/* disable INTST1 interrupt */
	SRIF1 = 0U;	/* clear INTSR1 interrupt flag */
	SRMK1 = 0U;	/* enable INTSR1 interrupt */
	SREIF1 = 0U;	/* clear INTSRE1 interrupt flag */
	SREMK1 = 0U;	/* enable INTSRE1 interrupt */
	SO0 |= _0004_SAU_CH2_DATA_OUTPUT_1;	/* output level normal */
	SOE0 |= _0004_SAU_CH2_OUTPUT_ENABLE;	/* enable UART1 output */
	SS0 |= _0008_SAU_CH3_START_TRG_ON | _0004_SAU_CH2_START_TRG_ON;	/* enable UART1 receive and transmit */
}
/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function stops the UART1 module operation.
**
**  Parameters:
**	None
**
**  Returns:
**	None
**
**-----------------------------------------------------------------------------
*/
void UART1_Stop(void)
{
	ST0 |= _0008_SAU_CH3_STOP_TRG_ON | _0004_SAU_CH2_STOP_TRG_ON;	/* disable UART1 receive and transmit */
	SOE0 &= ~_0004_SAU_CH2_OUTPUT_ENABLE;	/* disable UART1 output */
	STMK1 = 1U;	/* disable INTST1 interrupt */
	STIF1 = 0U;	/* clear INTST1 interrupt flag */
	SRMK1 = 1U;	/* disable INTSR1 interrupt */
	SRIF1 = 0U;	/* clear INTSR1 interrupt flag */
	SREMK1 = 1U;	/* disable INTSRE1 interrupt */
	SREIF1 = 0U;	/* clear INTSRE1 interrupt flag */
}
/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function receives UART1 data.
**
**  Parameters:
**	rxbuf: receive buffer pointer
**	rxnum: buffer size
**
**  Returns:
**	MD_OK
**	MD_ARGERROR
**
**-----------------------------------------------------------------------------
*/
MD_STATUS UART1_ReceiveData(uint8_t *rxbuf, uint16_t rxnum)
{
	MD_STATUS status = MD_OK;

	if (rxnum < 1U)
	{
		status = MD_ARGERROR;
	}
	else
	{
		gUart1RxCnt = 0U;
		gUart1RxLen = rxnum;
		gpUart1RxAddress = rxbuf;		
		if( gUart1RxLen > USART_IF_BUFFER)
			gUart1RxLen = USART_IF_BUFFER;

	}

	return (status);
}
/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function sends UART1 data.
**
**  Parameters:
**	txbuf: transfer buffer pointer
**	txnum: buffer size
**
**  Returns:
**	MD_OK
**	MD_ARGERROR
**
**-----------------------------------------------------------------------------
*/
MD_STATUS UART1_SendData(uint8_t* txbuf, uint16_t txnum)
{
	MD_STATUS status = MD_OK;

	if (txnum < 1U)
	{
		status = MD_ARGERROR;
	}
	else
	{
		gpUart1TxAddress = txbuf;
		gUart1TxCnt = txnum;
		STMK1 = 1U;	/* disable INTST1 interrupt */
		TXD1 = *gpUart1TxAddress;
		gpUart1TxAddress++;
		gUart1TxCnt--;
		STMK1 = 0U;	/* enable INTST1 interrupt */
	}

	return (status);
}
/**
  * @brief  Ensures that the CS is not asserted before the SPI finish 
			transmitting data
  * @param  None
  * @retval None
  */
static void UartWaitForTxFinish(void)
{
    /* Loop until the end of SPI transmit */
	while(gUart1TxCnt)	
	{
		;
	}
}

/**
  * @brief  Configure the USART and specifies the baudrate
  * @caller main
  * @param None
  * @retval None
  */
void DispatcherUARTConfigure(unsigned long ulBaudRate)
{    
 	/* Start Uart Module */
	UART1_Start();
	UART1_ReceiveData((uint8_t *)g_ucUARTBuffer, USART_IF_BUFFER);		
}

/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function is sends data out of the serial port.
**
**  Parameters:
**	inBuff   : A pointer to data to be sent on the UART
**  usLenght : The lenght of the data
**  Returns:
**	None
**
**-----------------------------------------------------------------------------
*/
void DispatcherUartSendPacket( uint8_t *inBuff, unsigned short usLength)
{
	UART1_Start();	
	UART1_SendData(inBuff, usLength);	
	UartWaitForTxFinish();	
    uart_have_cmd = 0x00; 	
}

